//-----------------------------------------------------------------------------
// Copyright 2018 Thiago Alves
// This file is part of the OpenPLC Software Stack.
//
// OpenPLC is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// OpenPLC is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with OpenPLC.  If not, see <http://www.gnu.org/licenses/>.
//------
//
// This is the main file for the OpenPLC Arduino firmware. It contains the a
// Modbus RTU library to communciate with OpenPLC host as a slave device.
//
// Thiago Alves, Aug 2018
//-----------------------------------------------------------------------------

/************************PINOUT CONFIGURATION*************************

Digital In: 22, 23, 24, 25, 26, 27, 28, 29,       (%IX100.0 - %IX100.7)
            30, 31, 32, 33, 34, 35, 36, 37        (%IX101.0 - %IX101.7)

			
Digital Out: 42, 41, 40, 5, 4, 3, 2, 49,          (%QX100.0 - %QX100.7)
             48, 47, 46, 45, 44, 43               (%QX101.0 - %QX101.5)

Analog In: A0, A1, A2, A3, A4, A5, A6, A7         (%IW100 - %IW107)

		   
Analog Out: 6, 7, 8, 9, 10, 11, 12, 13            (%QW100 - %QW107)
			
*********************************************************************/

#include <Arduino.h>
#include "Modbus.h"
#include "ModbusSerial.h"

//ModBus Port information
#define BAUD        9600
#define ID          1
#define TXPIN       -1

//Define the number of registers, inputs and coils to be created
#define NUM_DISCRETE_INPUT      16
#define NUM_INPUT_REGISTERS     8
// Neu
#define NUM_COILS               14

#define NUM_HOLDING_REGISTERS   8

//Create the I/O pin masks
// Digital IN
uint8_t pinMask_DIN[] = {22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37};
// Analog IN
uint8_t pinMask_AIN[] = {A0, A1, A2, A3, A4, A5, A6, A7};
// Relais OUT
uint8_t pinMask_DOUT[] = {42, 41, 40, 5, 4, 3, 2, 49, 48, 47, 46, 45, 44, 43};
// Analog OUT
uint8_t pinMask_AOUT[] = {6, 7, 8, 9, 10, 11, 12, 13};

//Modbus Object
ModbusSerial modbus;

/*
 * 
analogReference()



    DEFAULT: the default analog reference of 5 volts (on 5V Arduino boards) or 3.3 volts (on 3.3V Arduino boards)

    INTERNAL: an built-in reference, equal to 1.1 volts on the ATmega168 or ATmega328P and 2.56 volts on the ATmega32U4 and ATmega8 (not available on the Arduino Mega)

    INTERNAL1V1: a built-in 1.1V reference (Arduino Mega only)

    INTERNAL2V56: a built-in 2.56V reference (Arduino Mega only)

    EXTERNAL: the voltage applied to the AREF pin (0 to 5V only) is used as the reference.


 * 
 */



void configurePins()
{
    for (int i = 0; i < NUM_DISCRETE_INPUT; i++)
    {
        pinMode(pinMask_DIN[i], INPUT);
    }
    
    for (int i = 0; i < NUM_INPUT_REGISTERS; i++)
    {
        pinMode(pinMask_AIN[i], INPUT);
    }
    
    for (int i = 0; i < NUM_COILS; i++)
    {
        pinMode(pinMask_DOUT[i], OUTPUT);
    }

    for (int i = 0; i < NUM_HOLDING_REGISTERS; i++)
    {
        pinMode(pinMask_AOUT[i], OUTPUT);
    }
    pinMode(TXPIN,OUTPUT);
}

void setup()
{
    //Setup board I/O
    configurePins();
    // analogReference(INTERNAL2V56); // 0..1023 0-2.56V
    
    //Config Modbus Serial (port, speed, rs485 tx pin)
    modbus.config(&Serial, BAUD, TXPIN);
    
    //Set the Slave ID
    modbus.setSlaveId(ID); 
    
    //Add all modbus registers
    for (int i = 0; i < NUM_DISCRETE_INPUT; ++i) 
    {
        modbus.addIsts(i);
    }
    for (int i = 0; i < NUM_INPUT_REGISTERS; ++i) 
    {
        modbus.addIreg(i);
    }
    for (int i = 0; i < NUM_COILS; ++i) 
    {
        modbus.addCoil(i);
    }
    for (int i = 0; i < NUM_HOLDING_REGISTERS; ++i) 
    {
        modbus.addHreg(i);
    }
}

void loop()
{
    //Run the main modbus task
    modbus.task();
    
    //Update modbus registers
    for (int i = 0; i < NUM_DISCRETE_INPUT; ++i) 
    {
        modbus.Ists(i, !digitalRead(pinMask_DIN[i]));
    }
    for (int i = 0; i < NUM_INPUT_REGISTERS; ++i) 
    {
        modbus.Ireg(i, (analogRead(pinMask_AIN[i]) * 32));
        // modbus.Ireg(i, (analogRead(pinMask_AIN[i])));		// 0..1023 0-2.56V
		
    }
    for (int i = 0; i < NUM_COILS; ++i) 
    {
        digitalWrite(pinMask_DOUT[i], modbus.Coil(i));
    }
    for (int i = 0; i < NUM_HOLDING_REGISTERS; ++i) 
    {
        analogWrite(pinMask_AOUT[i], (modbus.Hreg(i) / 256));
    }
}
